'''License

Copyright © 2018, Numato Systems Private Limited. All rights reserved.

This software including all supplied files, Intellectual Property, know-how
or part of thereof as applicable (collectively called SOFTWARE) in source
and/or binary form with accompanying documentation is licensed to you by
Numato Systems Private Limited (LICENSOR) subject to the following conditions.

1. This license permits perpetual use of the SOFTWARE if all conditions in this
   license are met. This license stands revoked in the event of breach of any
   of the conditions.    
2. You may use, modify, copy the SOFTWARE within your organization. This
   SOFTWARE shall not be transferred to third parties in any form except
   fully compiled binary form as part of your final application.
3. This SOFTWARE is licensed only to be used in connection with/executed on
   supported products manufactured by Numato Systems Private Limited.
   Using/executing this SOFTWARE on/in connection with custom or third party
   hardware without the LICENSOR's prior written permission is expressly
   prohibited.
4. You may not download or otherwise secure a copy of this SOFTWARE for the
   purpose of competing with Numato Systems Private Limited or subsidiaries in
   any way such as but not limited to sharing the SOFTWARE with competitors, 
   reverse engineering etc... You may not do so even if you have no gain 
   financial or otherwise from such action.
5. DISCLAIMER
5.1. USING THIS SOFTWARE IS VOLUNTARY AND OPTIONAL. NO PART OF THIS SOFTWARE
     CONSTITUTE A PRODUCT OR PART OF PRODUCT SOLD BY THE LICENSOR.
5.2. THIS SOFTWARE AND DOCUMENTATION ARE PROVIDED “AS IS” WITH ALL FAULTS,
     DEFECTS AND ERRORS AND WITHOUT WARRANTY OF ANY KIND.
5.3. THE LICENSOR DISCLAIMS ALL WARRANTIES EITHER EXPRESS OR IMPLIED, INCLUDING
     WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY OR FITNESS FOR ANY
     PURPOSE.
5.4. IN NO EVENT, SHALL THE LICENSOR, IT’S PARTNERS OR DISTRIBUTORS BE LIABLE OR
     OBLIGATED FOR ANY DAMAGES, EXPENSES, COSTS, LOSS OF MONEY, LOSS OF TANGIBLE
     OR INTANGIBLE ASSETS DIRECT OR INDIRECT UNDER ANY LEGAL ARGUMENT SUCH AS BUT
     NOT LIMITED TO CONTRACT, NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH
     OF WARRANTY OR ANY OTHER SIMILAR LEGAL DEFINITION.'''

''' Python code demonstrating features of Numato Lab 64 Channel 
    USB Relay Module.
    
    Usage: python scriptname.js <PORT>
    E.g.: python 64usbrelay.js COM18 '''
###############################################################################
#                                                                             #
#   Prerequisites                                                             #
#   ------------                                                              #
#   Python version 3.x                                                        #
#   pip version 6.x                                                           #
#   pySerial version 3.x                                                      #
#                                                                             #
###############################################################################
import sys
import serial
import time
import re

###############################################################################
#                      Module related misc. functions                         #
###############################################################################

'''Retrieve and print FW version'''
def getFWVersion():
    serPort.flushInput()
    #Send ver command. Note the newline appended to the command.
    serPort.write(b"ver\r")
    time.sleep(1)

    
    #Read the response and print the firmware version
    response = serPort.read(25)
    return response[5:-1]
    
'''Retrieve and print Module ID'''    
def getModuleid():
    serPort.flushInput()
    #Get Current Module ID
    serPort.write(b"id get\r")
    time.sleep(1)
    response = serPort.read(25).decode()
    return response[8:-1]

'''Sets new Module ID'''
def setModuleid():
    #Get new Module ID form user 
    while True:
        new_id = input("Enter new ID (8 alphanumeric characters only): ")
        if len(new_id) == 8:
            break
            
    serPort.write(b"id set " + new_id.encode("ascii")+ b"\r")
    return ("\nModule ID Set Successfully\n")

'''Reset the module'''
def reset():
    serPort.write(b"reset\r");
    print("\nReset successfully\n");

###############################################################################
#                            Relay operations                                 #
###############################################################################

'''Retrieve and print relay status'''
def getRelayStatus():
    serPort.flushInput()
    while True:

        #Get relay index/number from the user
        relay_number = ("%02d" % int(input("Please enter relay number (0 to 63): ")))
        if relay_number < "64":
            break

    #Send command
    serPort.write(b"relay read " + relay_number.encode("ascii") + b"\r")
    time.sleep(1)
    
    #Read and process response from the device
    response = serPort.read(20)
    return response[15:], relay_number

'''Relay ON'''
def relayOn():
    while True:
        #Get relay index/number from the user
        relay_number = ("%02d" % int(input("Please enter relay number (0 to 63): ")))
        if relay_number < "64":
            break
    serPort.write(("relay on " + relay_number + "\r").encode("ascii"))
    print("\n")
    time.sleep(1)
    #Empty the input buffer
    serPort.flushInput()

'''Relay OFF''' 
def relayOff():
    while True:
        #Get relay index/number from the user
        relay_number = ("%02d" % int(input("Please enter relay number (0 to 63): ")))
        if relay_number < "64":
            break
    serPort.write(("relay off " + relay_number + "\r").encode("ascii"))
    print("\n")
    time.sleep(1)
    #Empty the input buffer
    serPort.flushInput()

'''Retrieve and print readall status'''
def readall():
    #Empty the input buffer
    serPort.flushInput()
    #Send Command 
    serPort.write(b"relay readall\r")
    time.sleep(1)

    #Read Relay Status
    response = serPort.read(35)
    return response[15:-1]

'''Write to all Relays at a time'''
def writeall():
    #Get new Relay status from user
    while True:
        value = (input("Enter Value to write(E.g.:1234f3a2cdef1234) : ").lower())
        if len(value) == 16:
            if re.match("^[a-f0-9]*$", value):
                break
    #Send Command 
    serPort.write(("relay writeall" + " " + value + "\r").encode("ascii"))
    #Empty the input buffer
    serPort.flushInput()

###############################################################################
#                           ADC/Analog operations                             #
###############################################################################

'''Retrieve and print ADC status'''
def getADCStatus():
    serPort.flushInput()
    #Get ADC index/number from the user
    while True:
        adc_number = input("Please enter ADC number (0 to 5): ")
        if int(adc_number) < 6:
            break

    #Send command
    serPort.write(b"adc read " +(adc_number.encode("ascii")) + b"\r")
    time.sleep(1)
    
    #Read and process response from the device
    response = serPort.read(25)
    return response[12:-1], adc_number

###############################################################################
#                             GPIO operations                                 #
###############################################################################

'''Retrieve and print GPIO status'''
def getGpioStatus():
    #Get GPIO index/number from the user
    while True:
        gpioNum= input("Please enter gpio number (0 to 7): ")
        if int(gpioNum) < 8:
            break

    #Send "gpio read" command
    serPort.write(b"gpio read "+ gpioNum.encode("ascii") + b"\r")
        
    time.sleep(1)

    #Read and process response from the device    
    response = serPort.read(17)

    return response[13:-1], gpioNum

'''Set GPIO high'''
def gpioSet():
    #Get GPIO index/number from the user
    while True:
        gpioNum= input("Please enter gpio number (0 to 7): ")
        if int(gpioNum) < 8:
            break

    #Send the command
    serPort.write(("gpio set "+ gpioNum + "\r").encode("ascii"))
    
    print("Command sent...")

'''Set GPIO low'''
def gpioClear():
    #Get GPIO index/number from the user
    while True:
        gpioNum= input("Please enter gpio number (0 to 7): ")
        if int(gpioNum) < 8:
            break

    #Send the command
    serPort.write(("gpio clear "+ gpioNum + "\r").encode("ascii"))
    
    print("Command sent...")

###############################################################################
#                                  Menus                                      #
###############################################################################

'''Display the main menu and returns user selection'''    
def getMenuSelection():
    #Initialize menu placeholder
    menu = {}

    #Build menu
    print("Please select from following menu options: \n")
    print("Menu:")
    menu['1']="Get Firmware Version" 
    menu['2']="Get Module ID"
    menu['3']="Set Module ID"
    menu['4']="Relay Status and Control"
    menu['5']="ADC Status"
    menu['6']="GPIO Status and control"
    menu['7']="Reset" 
    menu['q']="Exit"

    #Print menu
    options = menu.keys()
    
    for entry in options: 
        print(entry, menu[entry])

    #Accept the user input and return
    return input("\nEnter selection:  ")

'''Display the relay functions menu and returns user selection'''
def RelayStatus_Control():
    #Initialize menu placeholder
    menu = {}
    
    #Build menu
    print("Please select from following menu options: \n")
    menu['1']="Individual Relay Status"
    menu['2']="Read all Relays"
    menu['3']="Individual Relay Control"
    menu['4']="Write all Relays"
    menu['b']="Back to main menu"
    menu['q']="Exit"


    #Print menu
    options = menu.keys()
    
    for entry in options: 
        print(entry, menu[entry])

    #Accept the user input and return
    return input("\nEnter selection:  ")

'''Display the relay control menu and returns user selection'''
def relaycontrol():
    #Initialize menu
    menu = {}
    
    #Build menu
    print("Please select from following menu options: \n")
    menu['1']="ON Relay"
    menu['2']="OFF Relay"
    menu['3']="Back to previous menu"
    menu['b']="Back to main menu"
    menu['q']="Exit"

    #Print menu
    options = menu.keys()
    
    for entry in options: 
        print(entry, menu[entry])

    #Accept the user input and return
    return input("\nEnter selection:  ")

'''Display the gpio functions menu and returns user selection'''
def GPIO():
    #Initialize menu 
    menu = {}
    
    #Build menu
    print("Please select from following menu options: \n")
    menu['1']="GPIO Status"
    menu['2']="GPIO Control"
    menu['b']="Back to main menu"
    menu['q']="Exit"

    #Print menu
    options = menu.keys()
    
    for entry in options: 
        print(entry, menu[entry])

    #Accept the user input and return
    return input("\nEnter selection:  ")    

'''Display the gpio control menu and returns user selection'''
def gpioControl():
    #Initialize menu placeholder
    menu = {}
    
    #Build menu
    print("Please select from following menu options: \n")
    menu['1']="Set GPIO"
    menu['2']="Clear GPIO"
    menu['3']="Back to previous menu"
    menu['b']="Back to main menu"
    menu['q']="Exit"

    #Print menu
    options = menu.keys()
    
    for entry in options: 
        print(entry, menu[entry])

    #Accept the user input and return
    return input("\nEnter selection:  ")

###############################################################################
#                                 Version related code                        #
###############################################################################
#This code works with python3 and above
#Check the python version
if sys.version_info[0] < 3:
    raise Exception("Python version 3.x required")

if (len(sys.argv) < 2):
	print ("Usage: scriptname.py <PORT>\nEg: 64usbrelay.py COM18")
	sys.exit(0)
else:
	port = sys.argv[1];
	
###############################################################################
#                       Serial port related methods                           #
###############################################################################

#Create Serialport object and Open port for communication    
serPort = serial.Serial(port, 19200, timeout = 1)

#Check the port
if serPort.is_open:
        print("\nConnected to ",port)
else:
        print("Check the device.")
        serPort.close()
        exit(0)

###############################################################################
#                       Main Application Code                                 #
###############################################################################
#Connected to device successfully. Display menu and handle user selection.   
while True:

    #Display menu and retrieve user selection
    selectedMenuItem = getMenuSelection()

    #Empty the input buffer
    serPort.flushInput()

    #Process the selected option
    if selectedMenuItem == '1': 
        print("\nCurrent Firmware Version: ", getFWVersion().decode())

    elif selectedMenuItem == '2': 
        print("\nModule ID: ",getModuleid())

    elif selectedMenuItem == '3': 
        print(setModuleid())

    elif selectedMenuItem == '4':
        while True:
                
            relay=RelayStatus_Control()
                
            #Process the selected option
            if relay == '1':
                print("\nRead Relay status\n")
                status, relay_number = getRelayStatus() 
                print("\nRelay #", relay_number, "status is", status.decode())
            elif relay == '2':         
                response = readall()
                print("\nReadall Status : ",response.decode())
            elif relay == '3':
                while True:
                    rlycntrl = relaycontrol()
                    if rlycntrl == '1':
                        relayOn()
                    elif rlycntrl == '2':
                        relayOff()
                    elif rlycntrl == '3':
                        break
                    elif rlycntrl.lower() == 'b':
                        break
                    elif rlycntrl.lower() == 'q':
                        print("\nExiting...")
                        exit(0)
                    else:
                        print("\n\nUnknown Option Selected! Please enter a valid option from menu below\n")
                if rlycntrl.lower() == 'b':
                    break
            elif relay == '4':
                writeall()
            elif relay.lower() == 'b':
                break
            elif relay.lower() == 'q':
                print("\nExiting...")
                exit(0)
            else:
                print("\n\nUnknown Option Selected! Please enter a valid option from menu below\n")
                    
    elif selectedMenuItem == '5':
        print("\nRead ADC status\n")
        status, adc_number = getADCStatus() 
        print("\nADC #", adc_number, "status is", status.decode())
        
    elif selectedMenuItem == '6':
        while True:

            #Empty the input buffer
            serPort.flushInput()

            gpio = GPIO()

            #Process the selected option
            if gpio == '1':
                print("\nRead GPIO status\n")
                status, gpio_number = getGpioStatus()
                print("\nGPIO #", gpio_number, "status is", status.decode())
            elif gpio == '2':
                while True:
                    gpiocntrl = gpioControl()
                    if gpiocntrl == '1':
                        gpioSet()
                    elif gpiocntrl == '2':
                        gpioClear()
                    elif gpiocntrl == '3':
                        break
                    elif gpiocntrl.lower() == 'b':
                        break
                    elif gpiocntrl.lower() =='q':
                        print("\nExiting...")
                        exit(0)
                    else:
                        print("\n\nUnknown Option Selected! Please enter a valid option from menu below\n")
                if gpiocntrl.lower() == 'b':
                    break
            elif gpio.lower() == 'b':
                break
            elif gpio.lower() == 'q':
                print("\nExiting...")
                exit(0)
            else:
                    print("\n\nUnknown Option Selected! Please enter a valid option from menu below\n")
    elif selectedMenuItem == '7':
        print(reset())
    elif selectedMenuItem.lower() == 'q': 
        print("\nExiting...")
        exit(0)

    else: 
            print("\n\nUnknown Option Selected! Please enter a valid option from menu below\n")




#Close the port
serPort.close()




