/*
## License

Copyright © 2017, Numato Systems Private Limited. All rights reserved.

This software including all supplied files, Intellectual Property, know-how
or part of thereof as applicable (collectively called SOFTWARE) in source
and/or binary form with accompanying documentation is licensed to you by
Numato Systems Private Limited (LICENSOR) subject to the following conditions.

1. This license permits perpetual use of the SOFTWARE if all conditions in this
   license are met. This license stands revoked in the event of breach of any
   of the conditions.    
2. You may use, modify, copy the SOFTWARE within your organization. This
   SOFTWARE shall not be transferred to third parties in any form except
   fully compiled binary form as part of your final application.
3. This SOFTWARE is licensed only to be used in connection with/executed on
   supported products manufactured by Numato Systems Private Limited.
   Using/executing this SOFTWARE on/in connection with custom or third party
   hardware without the LICENSOR's prior written permission is expressly
   prohibited.
4. You may not download or otherwise secure a copy of this SOFTWARE for the
   purpose of competing with Numato Systems Private Limited or subsidiaries in
   any way such as but not limited to sharing the SOFTWARE with competitors, 
   reverse engineering etc... You may not do so even if you have no gain 
   financial or otherwise from such action.
5. DISCLAIMER
5.1. USING THIS SOFTWARE IS VOLUNTARY AND OPTIONAL. NO PART OF THIS SOFTWARE
     CONSTITUTE A PRODUCT OR PART OF PRODUCT SOLD BY THE LICENSOR.
5.2. THIS SOFTWARE AND DOCUMENTATION ARE PROVIDED “AS IS” WITH ALL FAULTS,
     DEFECTS AND ERRORS AND WITHOUT WARRANTY OF ANY KIND.
5.3. THE LICENSOR DISCLAIMS ALL WARRANTIES EITHER EXPRESS OR IMPLIED, INCLUDING
     WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY OR FITNESS FOR ANY
     PURPOSE.
5.4. IN NO EVENT, SHALL THE LICENSOR, IT’S PARTNERS OR DISTRIBUTORS BE LIABLE OR
     OBLIGATED FOR ANY DAMAGES, EXPENSES, COSTS, LOSS OF MONEY, LOSS OF TANGIBLE
     OR INTANGIBLE ASSETS DIRECT OR INDIRECT UNDER ANY LEGAL ARGUMENT SUCH AS BUT
     NOT LIMITED TO CONTRACT, NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH
     OF WARRANTY OR ANY OTHER SIMILAR LEGAL DEFINITION.
*/

/*
    Node.JS (Javascript) code demonstrating features of Numato Lab 64 Channel 
    USB Relay Module.
    
    Usage: node scriptname.js <PORT>
    E.g.: node 64usbrelay.js COM15
*/

/*
 Prerequisites
 -------------
 install serialport
 install readline
*/

var SerialPort = require('serialport');
const readline = require('readline');

const rl = readline.createInterface({
        input: process.stdin, 
        output: process.stdout
});

/* Check and parse arguments */
if(process.argv.length !=3) {
    console.log("\nUsage: node scriptname.js <PORT>\nE.g.:node 64usbrelay.js COM15\n");
    process.exit(code=0);
}
else {
    var port = process.argv[2];
}

/*****************************************************************************/
/*                    Serial port related methods                            */
/*****************************************************************************/

/* Create Serialport object */
var portObj = new SerialPort(port, {
        baudRate: 19200,
        autoOpen: false
});

/*  
 The serial port object's data event. This event will be 
 triggered every time data is received from the device.
*/
portObj.on('data', function(data){
    console.log('--------------------------');
    data = String(data.slice(0,-1));
    console.log(data.substring(data.indexOf("\r") + 1));
});

/* Open Port for communication */
portObj.open(function (error){
    if (error) {
        console.log('Failed to open port: ' + error);
        process.exit(code=0);
    } 
    else {
        main();
    }
});

/*****************************************************************************/
/*                       Module related misc. functions                      */
/*****************************************************************************/

/* Sets new Module ID */
function set_id() {
    rl.question("Enter new Module ID : ", (new_id) => {
        if (new_id.length === 8) {
            portObj.write('id set ' + new_id + '\r');
            console.log('\nModule ID Set successfully');
            setTimeout(()=> {
                main();
            },1000);
        }
        else {
            set_id();
        }   
    });
}

/* Reset the module */
function Reset() {
    portObj.write('reset\r');
    console.log('\nReset successfully');
    setTimeout(()=> {
        main();
    },1000);
}

/*****************************************************************************/
/*                             Relay operations                              */
/*****************************************************************************/

/* Retrieve and print relay status */
function get_relaystatus() {
    rl.question("Please enter relay number (0 to 63): ", (relay_num) => {
        if( relay_num < 64 ) {
            if( relay_num <10 ) {
                relay_num = ("0" + relay_num).slice(-2);
            }
            portObj.write('relay read ' + relay_num + '\r');
            console.log('\nRelay #' + relay_num + ' status');
            setTimeout(()=> {
                relay_functions();
            },1000);
        }
        else {
            get_relaystatus();
        }
    });
}

/* Relay ON */
function relay_on() {
    rl.question("Please enter relay number (0 to 63): ", (relay_num) => {
        if( relay_num < 64 ) {
            if( relay_num <10 ) {
                relay_num = ("0" + relay_num).slice(-2);
            }
            portObj.write('relay on ' + relay_num + '\r');
            setTimeout(()=> {
                relay_control();
            },1000);
        }
        else {
            relay_on();
        }
    });
}

/* Relay OFF */
function relay_off() {
    rl.question("Please enter relay number (0 to 63): ", (relay_num) => {
        if( relay_num < 64 ) {
            if( relay_num <10 ) {
                relay_num = ("0" + relay_num).slice(-2);
            }
            portObj.write('relay off ' + relay_num + '\r');
            setTimeout(()=> {
                relay_control();
            },1000);
        }
        else {
            relay_off();
        }
    });
}

/* Retrieve and print readall status */
function readall() {
    portObj.write('relay readall\r');
    console.log("\nReadall Status");
    setTimeout(()=> {
        relay_functions();
    },1000);
}

/* Write to all relays at a time */
function writeall() {
    rl.question("Enter Data to write(E.g. : 1234f3a2cdef1234): ", (data) => {
        if( data.length === 16 ) {
            if(data.match(/^[0-9a-fA-F]+$/)) {
                portObj.write('relay writeall ' + data.toLowerCase() + '\r');
                setTimeout(()=>{
                    relay_functions();
                },1000);
            }
            else {
                writeall();
            }
        }
        else {
            writeall();
        }
    });
}

/*****************************************************************************/
/*                            ADC/Analog operations                          */
/*****************************************************************************/

/* Retrieve and print ADC status */
function adc_status() {
    rl.question("Please enter ADC number (0 to 5): ", (adc_number) => {
        if (adc_number < 6) {
            portObj.write('adc read '+ adc_number + '\r');
            console.log('\nADC #' + adc_number + ' status');
            setTimeout(()=> {
                main();
            },1000);
        }
        else {
            adc_status();
        }
    });
}

/*****************************************************************************/
/*                                GPIO operations                            */
/*****************************************************************************/

/* Retrieve and print GPIO status */
function get_gpiostatus() {
    rl.question("Please enter gpio number (0 to 7): ", (gpio_num) => {
        if( gpio_num < 8 ) {
            portObj.write('gpio read ' + gpio_num + '\r');
            console.log('\nGPIO #' + gpio_num + ' status');
            setTimeout(()=> {
                gpio_functions();
            },1000);
        }
        else {
            get_gpiostatus();
        }
    });
}

/* Set GPIO high */
function set_gpio() {
    rl.question("Please enter gpio number (0 to 7): ", (gpio_num) => {
        if( gpio_num < 8 ) {
            portObj.write('gpio set ' + gpio_num + '\r');
            console.log('\nSet successfully');
            setTimeout(()=> {
                gpio_control();
            },1000);
        }
        else {
            set_gpio();
        }
    });
}

/* Set GPIO low */
function clear_gpio() {
    rl.question("Please enter gpio number (0 to 7): ", (gpio_num) => {
        if (gpio_num < 8) {
            portObj.write('gpio clear ' + gpio_num + '\r');
            console.log('\nSet successfully');
            setTimeout(()=> {
                gpio_control();
            },1000);
        }
        else {
            clear_gpio();
        }
    });
}

/*****************************************************************************/
/*                                    Menus                                  */
/*****************************************************************************/

/* Display the relay functions menu and returns user selection */
function relay_functions() {
    var main_menu = [{"1":"Individual Relay Status"},{"2":"Read all Relays"},{"3":"Individual Relay Control"},{"4":"Write all relays"},{"b":"Back to main menu"},{"q":"Exit"}];
    for (i in main_menu){
        for (key in main_menu[i]){
            console.log( key + ": " + main_menu[i][key]);
        }
    }
    
    rl.question("please select from the options: ", (relay) => {
        if (relay === '1') {
            get_relaystatus();
        }
        else if (relay === '2') {
            readall();
        }
        else if (relay === '3') {
            relay_control();
        }
        else if (relay === '4') {
            writeall();
        }
        else if( relay.toLowerCase() ==='b' ) {
            main();
        }
        else if( relay.toLowerCase() ==='q' ) {
            console.log("\nExiting...");
            process.exit(code=0);
        }
        else {
            console.log("\nUnknown Option Selected! Please enter a valid option from menu below\n");            
            relay_functions();
        }
    });
}

/* Display the relay control menu and returns user selection*/
function relay_control() {
    var main_menu = [{"1":"Relay ON"},{"2":"Relay OFF"},{"3":"Back to previous menu"},{"b":"Back to main menu"},{"q":"Exit"}];
    for (i in main_menu){
        for (key in main_menu[i]){
            console.log( key + ": " + main_menu[i][key]);
        }
    }
    
    rl.question("please select from the options: ", (relay_cntrl) => {
        if (relay_cntrl === '1') {
            relay_on();
        }
        else if (relay_cntrl === '2') {
            relay_off();
        }
        else if (relay_cntrl === '3') {
            relay_functions();
        }
        else if (relay_cntrl.toLowerCase() === 'b') {
            main();
        }
        else if (relay_cntrl.toLowerCase() === 'q') {
            console.log("\nExiting...");
            process.exit(code=0);
        }
        else {
            console.log("\nUnknown Option Selected! Please enter a valid option from menu below\n");
            relay_control();
        }
    });
}

/* Display the gpio functions menu and returns user selection */
function gpio_functions() {
    var main_menu = [{"1":"GPIO Status"},{"2":"GPIO Control"},{"b":"Back to main menu"},{"q":"Exit"}];
    for (i in main_menu){
        for (key in main_menu[i]){
            console.log( key + ": " + main_menu[i][key]);
        }
    }
    
    rl.question("please select from the options: ", (gpio) => {
        if( gpio ==='1' ) {
            get_gpiostatus();
        }
        else if( gpio ==='2' ) {
            gpio_control();
        }
        else if( gpio.toLowerCase() ==='b' ) {
            main();
        }
        else if( gpio.toLowerCase() ==='q' ) {
            console.log("\nExiting...");
            process.exit(code=0);
        }
        else {
            console.log("\nUnknown Option Selected! Please enter a valid option from menu below\n");            
            gpio_functions();
        }
    });
}

/* Display the gpio control menu and returns user selection */
function gpio_control() {
    var main_menu = [{"1":"Set GPIO"},{"2":"Clear GPIO"},{"3":"Back to previous menu"},{"b":"Back to main menu"},{"q":"Exit"}];
    for (i in main_menu){
        for (key in main_menu[i]){
            console.log( key + ": " + main_menu[i][key]);
        }
    }
    
    rl.question("please select from the options: ", (gpio_cntrl) => {
        if (gpio_cntrl === '1') {
            set_gpio();
        }
        else if (gpio_cntrl === '2') {
            clear_gpio();
        }
        else if (gpio_cntrl === '3') {
            gpio_functions();
        }
        else if (gpio_cntrl.toLowerCase() === 'b') {
            main();
        }
        else if (gpio_cntrl.toLowerCase() === 'q') {
            console.log("\nExiting...");
            process.exit(code=0);
        }
        else {
            console.log("\nUnknown Option Selected! Please enter a valid option from menu below\n");
            gpio_control();
        }
    });
}

/*****************************************************************************/
/*                                    main                                   */
/*****************************************************************************/
function main() {
    var main_menu = [{"1":"Get Firmware Version"},{"2":"Get Module ID"},{"3":"Set Module ID"},{"4":"Relay Status and Control"},{"5":"ADC Status"},{"6":"GPIO Status and Control"},{"7":"Reset"},{"q":"Exit"}];
    console.log("Menu\n-----\n");
    for (i in main_menu){
        for (key in main_menu[i]){
            console.log( key + ": " + main_menu[i][key]);
        }
    }
    
    rl.question("please select from the options: ", (selection) => {
        if( selection==='1' ) {
            portObj.write('ver\r');
            console.log("\nVersion");
            setTimeout(()=> {
                main();
            },1000);
        }
        else if( selection==='2' ) {
            portObj.write('id get\r');
            console.log("\nModule ID");
            setTimeout(()=> {
                main();
            },1000);
        }
        else if( selection==='3' ) {
            set_id();
        }
        else if( selection==='4' ) {
            relay_functions();
        }
        else if( selection==='5' ) {
            adc_status();
        }
        else if( selection==='6' ) {
            gpio_functions();
        }
        else if( selection==='7' ) {
            Reset();
        }
        else if( selection.toLowerCase()==='q' ) {
            console.log("\nExiting...");
            process.exit(code=0);
        }
        else {
            console.log("\nUnknown Option Selected! Please enter a valid option from menu below\n");
            main();         
        }
    });
}